/* 
 * sigwaitinfo.c
 *
 * Передає сам собі кілька попередньо заблокованих сигналів реального
 * часу, після чого створює додатковий потік, який забезпечує обробку
 * цих сигналів у синхронному режимі.
 * Демонструє спосіб організації синхронної обробки сигналів у
 * багатопотоковій програмі. Застосовує функції pthread_sigmask(),
 * sigaction(), sigqueue(), sigwaitinfo().
 *
 */

#include <assert.h>
#include <errno.h>
#include <pthread.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* Номери сигналів, які передаються */
#define MYSIG_COUNT SIGRTMIN
#define MYSIG_STOP  MYSIG_COUNT + 1

void dummy_sig_handler(int signum, siginfo_t *info, void *context);
void *sig_thread(void *arg);


/* Фіктивний обробник сигналів MYSIG_COUNT, MYSIG_STOP.
   Потрібен тільки для того, щоб установити в диспозиції цих сигналів
   прапорець SA_SIGINFO і, тим самим, ґарантувати їх доставку і
   забезпечити можливість отримувати разом із ними додаткову інформацію. */
void dummy_sig_handler(int signum, siginfo_t *info, void *context)
{
}

/* Функція потоку, виділеного для обробки сигналів MYSIG_COUNT, MYSIG_STOP.
   Обробка сигналів виконується синхронним способом, тому ніяких обмежень
   на вибір функцій немає. */
void *sig_thread(void *arg)
{
        int signum;
        siginfo_t info;

        while(1) {
                /* Чекає сигнали, задані набором (sigset_t *) arg.
                   Номер отриманого сигналу зберігає в змінній *arg,
                   додаткову інформацію про нього - в структурі info. */
                signum = sigwaitinfo((sigset_t *) arg, &info);
                if (signum == -1) {
                        if (errno == EINTR)
                                continue;
                        else {
                                perror("sigwaitinfo()");
                                return (void *) 1;
                        }
                } else if (signum == MYSIG_COUNT) {
                        printf("Отримано сигнал MYSIG_COUNT"
                                " з рядком: %s.\n",
                                (char *) info.si_value.sival_ptr);
                } else if (signum == MYSIG_STOP) {
                        printf("Отримано сигнал MYSIG_STOP."
                                " Робота завершена.\n");
                        return (void *) 0;            
                } else
                        printf("Отримано сигнал із номером"
                                " %d.\n", signum);
        }
        return (void *) 1;
}

int main()
{
        sigset_t set;
        struct sigaction act;
        union sigval val;
        pthread_t tid;
        int errnum;

        /* Ініціалізує порожній набір сигналів. */
        if (sigemptyset(&set) != 0) {
                perror("sigemptyset()");
                exit(EXIT_FAILURE);
        }
        /* Додає в набір сигнал MYSIG_COUNT. */
        if (sigaddset(&set, MYSIG_COUNT) != 0) {
                perror("sigaddset()");
                exit(EXIT_FAILURE);
        }
        /* Додає в набір сигнал MYSIG_STOP. */
        if (sigaddset(&set, MYSIG_STOP) != 0) {
                perror("sigaddset()");
                exit(EXIT_FAILURE);
        }
        /* Блокує сигнали MYSIG_COUNT, MYSIG_STOP. */
        errnum = pthread_sigmask(SIG_SETMASK, &set, NULL);
        if (errnum != 0) {
                fprintf(stderr, "pthread_sigmask(): %s\n",
                                        strerror(errnum));
                exit(EXIT_FAILURE);
        }

        /* Реєструє фіктивний обробник сигналів MYSIG_COUNT, MYSIG_STOP. 
           Це потрібно тільки для того, щоб установити в диспозиції цих
           сигналів прапорець SA_SIGINFO і, тим самим, ґарантувати їх
           доставку і забезпечити можливість отримувати разом із ними
           додаткову інформацію. */
        memset(&act, 0, sizeof(act));
        act.sa_flags = SA_SIGINFO;
        act.sa_sigaction = dummy_sig_handler;
        if ((sigaction(MYSIG_COUNT, &act, NULL) != 0) 
                || (sigaction(MYSIG_STOP, &act, NULL) != 0)) {
                perror("sigaction()");
                exit(EXIT_FAILURE);
        }

        /* Передає сам собі три сигнали MYSIG_COUNT і один сигнал
           MYSIG_STOP. Оскільки ці сигнали заблоковані, вони будуть
           чекати обробки. */
        val.sival_ptr = "Один";
        if (sigqueue(getpid(), MYSIG_COUNT, val) != 0) {
                perror("sigqueue()");
                exit(EXIT_FAILURE);
        }
        val.sival_ptr = "Два";
        if (sigqueue(getpid(), MYSIG_COUNT, val) != 0) {
                perror("sigqueue()");
                exit(EXIT_FAILURE);
        }
        val.sival_ptr = "Три";
        if (sigqueue(getpid(), MYSIG_COUNT, val) != 0) {
                perror("sigqueue()");
                exit(EXIT_FAILURE);
        }
        val.sival_ptr = NULL;
        if (sigqueue(getpid(), MYSIG_STOP, val) != 0) {
                perror("sigqueue()");
                exit(EXIT_FAILURE);
        }

        /* Створює потік, призначений для обробки сигналів MYSIG_COUNT,
           MYSIG_STOP. */
        errnum = pthread_create(&tid, NULL, sig_thread, &set);
        if (errnum != 0) {
                fprintf(stderr, "pthread_create(): %s\n",
                                                strerror(errnum));
                exit(EXIT_FAILURE);
        }
        /* Чекає завершення дочірнього потоку. */
        errnum = pthread_join(tid, NULL);
        if (errnum != 0) {
                fprintf(stderr, "pthread_join(): %s\n", strerror(errnum));
                exit(EXIT_FAILURE);
        }

        /* Завершує роботу. */
        exit(EXIT_SUCCESS);
}
